/**
 * Parallax Engine
 * Smooth parallax scrolling for multiple layers
 * Lanka Matha Tours - About Page Advanced
 */

class ParallaxEngine {
    constructor() {
        this.elements = [];
        this.rafId = null;
        this.lastScrollY = window.pageYOffset;
        this.currentScrollY = this.lastScrollY;
        this.smoothFactor = 0.1; // Lerp smoothing factor

        this.init();
    }

    init() {
        this.findParallaxElements();
        if (this.elements.length > 0) {
            this.startAnimation();
            this.handleResize();
        }
    }

    findParallaxElements() {
        const parallaxElements = document.querySelectorAll('[data-parallax]');

        parallaxElements.forEach(element => {
            const speed = parseFloat(element.getAttribute('data-parallax')) || 0.5;
            this.elements.push({
                element: element,
                speed: speed,
                initialTop: element.getBoundingClientRect().top + window.pageYOffset
            });
        });
    }

    lerp(start, end, factor) {
        return start + (end - start) * factor;
    }

    updateParallax() {
        // Smooth scroll with lerp
        this.currentScrollY = this.lerp(
            this.currentScrollY,
            window.pageYOffset,
            this.smoothFactor
        );

        this.elements.forEach(item => {
            const elementTop = item.initialTop;
            const scrolled = this.currentScrollY;
            const relativeScroll = scrolled - elementTop + window.innerHeight;

            // Only apply parallax if element is in viewport range
            if (relativeScroll > -window.innerHeight && relativeScroll < window.innerHeight * 2) {
                const offset = relativeScroll * item.speed * -1;
                item.element.style.transform = `translateY(${offset}px)`;
            }
        });

        this.lastScrollY = this.currentScrollY;
    }

    animate() {
        this.updateParallax();
        this.rafId = requestAnimationFrame(() => this.animate());
    }

    startAnimation() {
        if (!this.rafId) {
            this.animate();
        }
    }

    stopAnimation() {
        if (this.rafId) {
            cancelAnimationFrame(this.rafId);
            this.rafId = null;
        }
    }

    handleResize() {
        let resizeTimeout;
        window.addEventListener('resize', () => {
            clearTimeout(resizeTimeout);
            resizeTimeout = setTimeout(() => {
                // Recalculate element positions on resize
                this.elements.forEach(item => {
                    item.initialTop = item.element.getBoundingClientRect().top + window.pageYOffset;
                });
            }, 250);
        });
    }

    // Add new parallax element dynamically
    addElement(element, speed = 0.5) {
        const item = {
            element: element,
            speed: speed,
            initialTop: element.getBoundingClientRect().top + window.pageYOffset
        };
        this.elements.push(item);
    }

    // Remove parallax element
    removeElement(element) {
        this.elements = this.elements.filter(item => item.element !== element);
    }
}

// Initialize on DOM ready
document.addEventListener('DOMContentLoaded', () => {
    // Check if user prefers reduced motion
    const prefersReducedMotion = window.matchMedia('(prefers-reduced-motion: reduce)').matches;

    if (!prefersReducedMotion) {
        const parallaxEngine = new ParallaxEngine();
        window.parallaxEngine = parallaxEngine;
    }
});

// Export for module usage
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ParallaxEngine;
}
