/**
 * Scroll Animation Controller
 * Detects when elements enter viewport and triggers animations
 * Lanka Matha Tours - About Page Advanced
 */

class ScrollAnimationController {
    constructor(options = {}) {
        this.options = {
            threshold: options.threshold || 0.15,
            rootMargin: options.rootMargin || '0px 0px -50px 0px',
            once: options.once !== undefined ? options.once : true
        };
        
        this.observer = null;
        this.init();
    }
    
    init() {
        // Check if Intersection Observer is supported
        if ('IntersectionObserver' in window) {
            this.createObserver();
            this.observeElements();
        } else {
            // Fallback for older browsers
            this.fallbackMode();
        }
    }
    
    createObserver() {
        const observerOptions = {
            threshold: this.options.threshold,
            rootMargin: this.options.rootMargin
        };
        
        this.observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    this.handleIntersect(entry.target);
                }
            });
        }, observerOptions);
    }
    
    observeElements() {
        // Observe elements with data-scroll-reveal attribute
        const elements = document.querySelectorAll('[data-scroll-reveal]');
        elements.forEach(element => {
            this.observer.observe(element);
        });
        
        // Observe elements with in-view class trigger
        const inViewElements = document.querySelectorAll('.story-content-card, .hex-item, .value-card, .feature-block');
        inViewElements.forEach(element => {
            this.observer.observe(element);
        });
    }
    
    handleIntersect(element) {
        // Add in-view class
        element.classList.add('in-view');
        
        // Apply custom delay if specified
        const delay = element.getAttribute('data-delay');
        if (delay) {
            element.style.transitionDelay = `${delay}s`;
            element.style.animationDelay = `${delay}s`;
        }
        
        // Trigger animation class for animated elements
        if (element.hasAttribute('data-scroll-reveal')) {
            element.classList.add('animated');
        }
        
        // Unobserve if once option is true
        if (this.options.once) {
            this.observer.unobserve(element);
        }
    }
    
    fallbackMode() {
        // Simple fallback: add in-view class to all elements
        const elements = document.querySelectorAll('[data-scroll-reveal], .story-content-card, .hex-item, .value-card');
        elements.forEach(element => {
            element.classList.add('in-view', 'animated');
        });
    }
    
    // Method to manually trigger animation on element
    triggerAnimation(element) {
        if (element) {
            this.handleIntersect(element);
        }
    }
    
    // Method to reset animation
    resetAnimation(element) {
        if (element) {
            element.classList.remove('in-view', 'animated');
            element.style.transitionDelay = '';
            element.style.animationDelay = '';
            
            if (this.observer) {
                this.observer.observe(element);
            }
        }
    }
}

// Initialize on DOM ready
document.addEventListener('DOMContentLoaded', () => {
    // Initialize scroll animation controller
    const scrollController = new ScrollAnimationController({
        threshold: 0.15,
        once: true
    });
    
    // Make it globally accessible for debugging
    window.scrollAnimationController = scrollController;
});

// Export for module usage
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ScrollAnimationController;
}
